<?php
// download_template.php
// Enhanced error logging and path handling

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Log function for debugging
function logDebug($message)
{
    error_log(date('Y-m-d H:i:s') . " - Download Template: " . $message);
}

// Ambil parameter dari URL
$templateName = $_GET['templateName'] ?? '';
logDebug("Request received for template: " . $templateName);

// Validasi input
if (empty($templateName)) {
    http_response_code(400);
    logDebug("Error: Template name is empty");
    echo json_encode(['error' => 'Template name is required']);
    exit;
}

// Sanitize filename untuk keamanan
$templateName = basename($templateName);
if (!preg_match('/^[a-zA-Z0-9_\-\.]+$/', $templateName)) {
    http_response_code(400);
    logDebug("Error: Invalid template name format");
    echo json_encode(['error' => 'Invalid template name format']);
    exit;
}

// Path ke folder template - dengan berbagai kemungkinan lokasi
$possiblePaths = [
    'assets/filetemplate/' . $templateName,
    '../assets/filetemplate/' . $templateName,
    './assets/filetemplate/' . $templateName,
    __DIR__ . '/assets/filetemplate/' . $templateName,
    __DIR__ . '/../assets/filetemplate/' . $templateName,
    __DIR__ . '/../../assets/filetemplate/' . $templateName,
    $_SERVER['DOCUMENT_ROOT'] . '/assets/filetemplate/' . $templateName,
    $_SERVER['DOCUMENT_ROOT'] . '/pinjamruang/assets/filetemplate/' . $templateName,
    $_SERVER['DOCUMENT_ROOT'] . '/bookingruang/assets/filetemplate/' . $templateName,
    // Tambahan path berdasarkan struktur project
    dirname(__DIR__) . '/assets/filetemplate/' . $templateName,
    dirname(dirname(__DIR__)) . '/assets/filetemplate/' . $templateName
];

$filePath = null;
$actualPath = '';

// Cari file di berbagai lokasi yang mungkin
foreach ($possiblePaths as $path) {
    logDebug("Checking path: " . $path);
    if (file_exists($path) && is_file($path)) {
        $filePath = $path;
        $actualPath = $path;
        logDebug("File found at: " . $path);
        break;
    }
}

// Debug: tampilkan semua path yang dicoba jika file tidak ditemukan
if (!$filePath) {
    logDebug("File not found. Tried paths:");
    foreach ($possiblePaths as $path) {
        logDebug("- " . $path . " (exists: " . (file_exists($path) ? 'YES' : 'NO') . ")");
    }

    // Tampilkan current directory untuk debugging
    logDebug("Current directory: " . getcwd());
    logDebug("Script directory: " . __DIR__);
    logDebug("Document root: " . $_SERVER['DOCUMENT_ROOT']);

    http_response_code(404);
    echo json_encode([
        'error' => 'Template not found',
        'template_name' => $templateName,
        'current_dir' => getcwd(),
        'script_dir' => __DIR__,
        'searched_paths' => $possiblePaths
    ]);
    exit;
}

// Validasi file
if (!is_readable($filePath)) {
    http_response_code(403);
    logDebug("File exists but not readable: " . $filePath);
    echo json_encode(['error' => 'Template file is not readable']);
    exit;
}

// Get file info
$fileInfo = pathinfo($filePath);
$fileName = $fileInfo['basename'];
$fileSize = filesize($filePath);

// Determine MIME type
$mimeType = 'application/octet-stream'; // Default fallback
if (function_exists('mime_content_type')) {
    $detectedMime = mime_content_type($filePath);
    if ($detectedMime !== false) {
        $mimeType = $detectedMime;
    }
} elseif (function_exists('finfo_open')) {
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $detectedMime = finfo_file($finfo, $filePath);
    finfo_close($finfo);
    if ($detectedMime !== false) {
        $mimeType = $detectedMime;
    }
}

// Set specific MIME type for common document formats
$extension = strtolower($fileInfo['extension'] ?? '');
switch ($extension) {
    case 'docx':
        $mimeType = 'application/vnd.openxmlformats-officedocument.wordprocessingml.document';
        break;
    case 'doc':
        $mimeType = 'application/msword';
        break;
    case 'pdf':
        $mimeType = 'application/pdf';
        break;
    case 'xlsx':
        $mimeType = 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet';
        break;
    case 'xls':
        $mimeType = 'application/vnd.ms-excel';
        break;
}

logDebug("Serving file: " . $fileName . " (" . $fileSize . " bytes) - MIME: " . $mimeType);

// Clear any previous output
if (ob_get_level()) {
    ob_end_clean();
}

// Set headers untuk download
header('Content-Type: ' . $mimeType);
header('Content-Disposition: attachment; filename="' . $fileName . '"');
header('Content-Length: ' . $fileSize);
header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
header('Pragma: public');
header('Accept-Ranges: bytes');
header('Expires: 0');

// Flush headers
if (ob_get_level()) {
    ob_end_flush();
}

// Stream file ke browser
$handle = fopen($filePath, 'rb');
if ($handle === false) {
    logDebug("Error: Failed to open file handle");
    http_response_code(500);
    echo json_encode(['error' => 'Failed to open template file']);
    exit;
}

// Stream file in chunks untuk file besar
$chunkSize = 8192; // 8KB chunks
while (!feof($handle)) {
    $chunk = fread($handle, $chunkSize);
    if ($chunk === false) {
        logDebug("Error: Failed to read file chunk");
        break;
    }
    echo $chunk;
    flush();
}

fclose($handle);
logDebug("File successfully served");
exit;
