<?php

namespace App\Controllers;

use CodeIgniter\Controller;
use CodeIgniter\RESTful\ResourceController;
use App\Models\UserModel;
use App\Models\RuanganModel;
use App\Models\PeminjamanModel;
use Exception;

class Frontend extends BaseController
{
    protected $UserModel;
    protected $RuanganModel;
    protected $PeminjamanModel;

    public function __construct()
    {
        $this->UserModel = new UserModel();
        $this->RuanganModel = new RuanganModel();
        $this->PeminjamanModel = new PeminjamanModel();
    }

    public function index(): string
    {
        $data['ruangan'] = $this->RuanganModel->findAll();

        return view('frontend/header')
        . view('beranda', $data)
            . view('frontend/footer');
    }

    public function ruangan(): string
    {
        $data['ruangan'] = $this->RuanganModel->findAll();

        return view('frontend/header')
        . view('ruangan',$data)
        . view('frontend/footer');
    }

    public function ruangan_detail($kode_ruang)
    {
        $ruangan = $this->RuanganModel->where('kode_ruang', $kode_ruang)->first(); // Fetch a single room by kode_ruang

        $data = [
            'kode_ruang' => $kode_ruang,
            'ruang' => $ruangan // Pass a single room to the view
        ];
        return view('frontend/header')
        . view('ruangan_detail', $data)
        . view('frontend/footer');
    }

    public function get_rooms()
    {
        $rooms = $this->RuanganModel->findAll();
        return $this->response->setJSON($rooms);
    }

    public function booking($kode_ruang)
    {
        // Ambil data ruang berdasarkan kode_ruang
        $ruangan = $this->RuanganModel->where('kode_ruang', $kode_ruang)->first(); // Fetch a single room by kode_ruang

        // Jika ruangan tidak ditemukan
        if (!$ruangan) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Ruang tidak ditemukan.');
        }

        // Siapkan data untuk dikirim ke view
        $data = [
            'kode_ruang' => $kode_ruang,
            'ruang' => $ruangan // Pass a single room to the view
        ];

        // Menampilkan header terlebih dahulu, kemudian halaman booking
        return view('frontend/header') . view('booking', $data);
    }

    public function booking_tambah()
    {
        // Verifikasi login dengan penanganan yang lebih baik
        $id_user = session()->get('id_user');

        if (!$id_user) {
            return redirect()->to(base_url('login'))->with('error', 'Anda harus login untuk melakukan peminjaman.');
        }

        // Periksa apakah request adalah POST
        if (!$this->request->getMethod() === 'post') {
            return redirect()->back()->with('error', 'Metode request tidak valid.');
        }

        // Validasi file yang diupload
        $validationRules = [
            'dok_peminjaman' => 'uploaded[dok_peminjaman]|mime_in[dok_peminjaman,application/pdf]|max_size[dok_peminjaman,2048]',
            'dok_ktm' => 'uploaded[dok_ktm]|mime_in[dok_ktm,image/png,image/jpg,image/jpeg]|max_size[dok_ktm,2048]'
        ];

        if (!$this->validate($validationRules)) {
            // Tampilkan pesan error validasi yang spesifik
            $validationErrors = $this->validator->getErrors();
            $errorMessage = implode(', ', $validationErrors);
            return redirect()->back()->withInput()->with('error', 'Validasi gagal: ' . $errorMessage);
        }

        // Ambil file yang diupload
        $dokPeminjaman = $this->request->getFile('dok_peminjaman');
        $dokKTM = $this->request->getFile('dok_ktm');

        // Pastikan folder penyimpanan ada
        if (!is_dir(FCPATH . 'assets/dokumen')) {
            mkdir(FCPATH . 'assets/dokumen', 0777, true);
        }
        if (!is_dir(FCPATH . 'assets/ktm')) {
            mkdir(FCPATH . 'assets/ktm', 0777, true);
        }

        // Upload dok_peminjaman
        $dokPeminjamanName = '';
        if ($dokPeminjaman->isValid() && !$dokPeminjaman->hasMoved()) {
            $dokPeminjamanName = $dokPeminjaman->getRandomName();
            try {
                $dokPeminjaman->move(FCPATH . 'assets/dokumen', $dokPeminjamanName);
            } catch (\Exception $e) {
                log_message('error', 'Gagal mengupload dokumen peminjaman: ' . $e->getMessage());
                return redirect()->back()->with('error', 'Gagal mengupload dokumen peminjaman: ' . $e->getMessage());
            }
        } else {
            return redirect()->back()->with('error', 'Dokumen peminjaman tidak valid.');
        }

        // Upload dok_ktm
        $dokKTMName = '';
        if ($dokKTM->isValid() && !$dokKTM->hasMoved()) {
            $dokKTMName = $dokKTM->getRandomName();
            try {
                $dokKTM->move(FCPATH . 'assets/ktm', $dokKTMName);
            } catch (\Exception $e) {
                log_message('error', 'Gagal mengupload dokumen KTM: ' . $e->getMessage());
                return redirect()->back()->with('error', 'Gagal mengupload dokumen KTM: ' . $e->getMessage());
            }
        } else {
            return redirect()->back()->with('error', 'Dokumen KTM tidak valid.');
        }

        // Persiapkan data untuk insert
        $dataInsert = [
            'id_user' => $id_user,
            'kode_ruang' => $this->request->getPost('kode_ruang'),
            'nama_ruang' => $this->request->getPost('nama_ruang'),
            'kapasitas' => $this->request->getPost('kapasitas'),
            'nama_kegiatan' => $this->request->getPost('nama_kegiatan'),
            'nama_organisasi' => $this->request->getPost('nama_organisasi'),
            'tanggal_mulai' => $this->request->getPost('tanggal_mulai'),
            'waktu_mulai' => $this->request->getPost('waktu_mulai'),
            'tanggal_selesai' => $this->request->getPost('tanggal_selesai'),
            'waktu_selesai' => $this->request->getPost('waktu_selesai'),
            'status' => $this->request->getPost('status') ?? 'Diproses',
            'dok_peminjaman' => 'assets/dokumen/' . $dokPeminjamanName,
            'dok_ktm' => 'assets/ktm/' . $dokKTMName
        ];

        // Log data yang akan diinsert untuk debugging
        log_message('info', 'Data yang akan diinsert: ' . json_encode($dataInsert));

        // Insert ke database dengan penanganan error
        try {
            $result = $this->PeminjamanModel->insert($dataInsert);

            // Periksa hasil insert
            if ($result) {
                log_message('info', 'Berhasil menyimpan data peminjaman');
                return redirect()->to(base_url('ruangan'))->with('success', 'Peminjaman berhasil ditambahkan');
            } else {
                log_message('error', 'Gagal menyimpan data ke database');
                return redirect()->back()->withInput()->with('error', 'Gagal menyimpan data ke database');
            }
        } catch (\Exception $e) {
            log_message('error', 'Error database: ' . $e->getMessage());
            return redirect()->back()->withInput()->with('error', 'Error database: ' . $e->getMessage());
        }
    }
    
    public function getPeminjamanByRoom($kodeRuang, $year, $month) {
    try {
        // Log input parameters for debugging
        log_message('debug', "getPeminjamanByRoom called with: kodeRuang=$kodeRuang, year=$year, month=$month");
        
        $model = new PeminjamanModel();
        
        // Format tanggal awal dan akhir bulan
        $awalBulan = "$year-$month-01";
        $akhirBulan = date('Y-m-t', strtotime($awalBulan));
        
        log_message('debug', "Interval tanggal: $awalBulan hingga $akhirBulan");
        
        // Query untuk mengambil data peminjaman tanpa memfilter status
        $peminjaman = $model->select('id_peminjaman, kode_ruang, nama_ruang, nama_kegiatan, nama_organisasi, tanggal_mulai, tanggal_selesai, waktu_mulai, waktu_selesai, status, catatan')
            ->where('kode_ruang', $kodeRuang)
            ->groupStart()
                ->groupStart()
                    ->where('tanggal_mulai >=', $awalBulan)
                    ->where('tanggal_mulai <=', $akhirBulan)
                ->groupEnd()
                ->orGroupStart()
                    ->where('tanggal_selesai >=', $awalBulan)
                    ->where('tanggal_selesai <=', $akhirBulan)
                ->groupEnd()
                ->orGroupStart()
                    ->where('tanggal_mulai <=', $awalBulan)
                    ->where('tanggal_selesai >=', $akhirBulan)
                ->groupEnd()
            ->groupEnd()
            ->get()
            ->getResultArray();
        
        // Log the SQL query for debugging
        log_message('debug', 'QUERY: ' . $model->getLastQuery()->getQuery());
        
        // Log the result count
        log_message('debug', 'Result count: ' . count($peminjaman));
        
        // Return a consistent response format
        return $this->response->setJSON([
            'status' => 'success', 
            'data' => $peminjaman
        ]);
    } catch (Exception $e) {
        log_message('error', 'Error getPeminjamanByRoom: ' . $e->getMessage());
        return $this->response->setJSON([
            'status' => 'error', 
            'message' => 'Terjadi kesalahan pada server: ' . $e->getMessage()
        ]);
    }
}

    public function cancel($id_peminjaman)
    {
        // Process the cancellation
        $model = new \App\Models\PeminjamanModel();
        $result = $model->update($id_peminjaman, ['status' => 'Ditolak']);

        // Set a flash message
        if ($result) {
            session()->setFlashdata('success', 'Peminjaman berhasil dibatalkan');
        } else {
            session()->setFlashdata('error', 'Gagal membatalkan peminjaman');
        }

        // Redirect back to booking history
        return redirect()->to('riwayatbooking');
    }




    public function login(): string
    {
        return view('login');
      
    }

    public function processLogin()
    {
        // Ambil data dari form
        $email = $this->request->getPost('email');
        $password = $this->request->getPost('password');

        // Validasi input
        if (!$email || !$password) {
            return redirect()->to(base_url('/login'))->with('error', 'Email dan password harus diisi');
        }

        // Cari user berdasarkan email
        $userModel = new \App\Models\UserModel();
        $user = $userModel->where('email', $email)->first();

        // Cek apakah user ditemukan dan password benar
        if (!$user || !password_verify($password, $user['password'])) {
            return redirect()->to(base_url('/login'))->with('error', 'Email atau password salah');
        }

        // Set session data dengan benar
        $sessionData = [
            'id_user' => $user['id_user'],
            'nama' => $user['nama_user'],
            'email' => $user['email'],
            'role' => $user['role'], // Pastikan kolom role ada di tabel user
            'logged_in' => true
        ];

        // Simpan session data
        session()->set($sessionData);

        // Regenerasi ID session untuk keamanan
        session()->regenerate();

        // Log aktivitas login berhasil
        log_message('info', 'User dengan ID: ' . $user['id_user'] . ' berhasil login dengan role ' . $user['role']);

        // Redirect berdasarkan role
        if ($user['role'] == 1) {
            // Role 1 - Admin ke dashboard
            return redirect()->to(base_url('/dashboard'))->with('success', 'Login berhasil');
        } else if ($user['role'] == 2) {
            // Role 2 - User ke beranda
            return redirect()->to(base_url('/index'))->with('success', 'Login berhasil');
        } else {
            // Fallback jika ada role lain
            return redirect()->to(base_url('/profile'))->with('success', 'Login berhasil');
        }
    }

    public function peminjamandetail($id_peminjaman)
    {
        $data['ruang'] = $this->RuanganModel->findAll(); // Jika hanya ingin semua data ruang
        $data['peminjaman'] = $this->PeminjamanModel->find($id_peminjaman); // Perbaiki ke model yang sesuai

        if (!$data['peminjaman']) {
            return redirect()->to('/riwayatbooking')->with('error', 'Data peminjaman tidak ditemukan');
        }

        return view('Frontend/header')
            . view('peminjamandetail', $data)
            . view('Frontend/footer');
    }



    public function logout()
    {
        session()->destroy();
        return redirect()->to('/login');
    }

    public function register(): string
    {
        return view('register');
    }

    public function processRegister()
    {
        $userModel = new UserModel();

        // Validasi input
        $validation = $this->validate([
            'nama_user'   => 'required|min_length[3]',
            'nomerinduk'  => 'required|is_unique[user.nomerinduk]',
            'no_hp'       => 'required|numeric',
            'email'       => 'required|valid_email|is_unique[user.email]',
            'password'    => 'required|min_length[6]'
        ]);

        if (!$validation) {
            return redirect()->to('/register')->withInput()->with('errors', $this->validator->getErrors());
        }

        // Debugging: Cek apakah data terkirim
        $data = [
            'nama_user'   => $this->request->getPost('nama_user'),
            'nomerinduk'  => $this->request->getPost('nomerinduk'),
            'no_hp'       => $this->request->getPost('no_hp'),
            'email'       => $this->request->getPost('email'),
            'password'    => password_hash($this->request->getPost('password'), PASSWORD_DEFAULT),
            'role'        => 2
        ];

        // Debug: Print data sebelum insert
        log_message('debug', 'Data yang akan disimpan: ' . json_encode($data));

        if (!$userModel->insert($data)) {
            log_message('error', 'Gagal menyimpan data: ' . json_encode($userModel->errors()));
            return redirect()->to('/register')->withInput()->with('errors', $userModel->errors());
        }

        return redirect()->to('/login')->with('success', 'Registrasi berhasil! Silakan login.');
    }

    public function tentang()
    {
        return view('frontend/header')
        . view('tentang')
        . view('frontend/footer');
    }

    public function downloadTemplate()
    {
        // Path file di folder public
        $filePath = FCPATH . 'assets/filetemplate/template_surat.docx';

        // Pastikan file ada sebelum diunduh
        if (file_exists($filePath)) {
            return $this->response->download($filePath, null);
        } else {
            // Jika file tidak ada, tampilkan pesan error
            return redirect()->to('/tentang')->with('error', 'File template surat tidak ditemukan.');
        }
    }

    public function riwayatbooking()
    {
        $session = session();
        $userId = $session->get('id_user'); // Ambil ID pengguna dari session

        $peminjamanModel = new PeminjamanModel();
        $data['peminjaman'] = $peminjamanModel->where('id_user', $userId)->findAll(); // Ambil data peminjaman berdasarkan user_id

        $data['session'] = $session; // Pastikan session tersedia di view
        return view('frontend/header', $data)
        . view('riwayatbooking', $data)
        . view('frontend/footer', $data);
    }

}
